/////////////////
// OS Includes
#include <fstream.h>
#include <iostream.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
//#include <unistd.h>
#include "getopt.h"

//////////////
// Includes
#include "ClauseList.h"
#include "Clause.h"
#include "Random.h"
#include "SATSolver.h"
#include "SATInstance.h"

/////////////
// Defines

void printHelpMessageAndExit() 
{
    cout << "==== relsat, version 2.00 (beta 2) ====" << endl;
    cout << "Usage: relsat [options] {filename}";
    cout << endl;
    cout << "Options include: " << endl;
    cout << " -l<int>          Learn order" << endl;
    cout << " -#{<int>|a|c}    {# of solutions | all | count}" << endl;
    cout << " -s<int>          Random # seed (> 0)" << endl;
    cout << " -r{<int>|n}      Restart interval {seconds | no restarts}" << endl;
    cout << " -f<float>        Fudge factor, in the range [0.0-1.0]" << endl;
    cout << " -t{<int>|n}      Timeout interval {seconds | no timeout}" << endl;
    cout << " -o{<filename>|n} Output instance {filename | no output instance}" << endl;
    cout << " -u{<int>|n}      Status update interval {seconds | no updates}" << endl;
    cout << " -p{0|1|2|3}      Pre-processing level" << endl;
    cout << " -c{0|1|2|3}      Post-processing level" << endl;
    cout << " -i{<int>|n}      Pre & post-processing iteration bound {bound | no bound}" << endl;
    cout << endl;
    cout << "Default options: " << endl;
    cout << "    -p1 -l3 -#1 -s1 -rn -f.9 -t43200 -u10 -o timeout.cnf -c3 -in" 
	 << endl;
    exit(1);
}

void main(int iArgc_, char* aArgv_[])
{
  if (iArgc_ < 2) {
    printHelpMessageAndExit();
  }
    
  int c;
  extern int optind;
  extern char *optarg;
  SATInstance xInstance;
  SATSolver xSATSolver(&xInstance);
  int iPreprocessLevel = 1;
  int iOutputPreprocessLevel = 3;
  int iIterationBound = -1;
  int iRandomSeed = 1;
  int iWork;
  float fWork;
  boolean bOutputInstance = 1;
  char aOutputInstance[512];
  boolean bNoTimeout = 0;
  strcpy(aOutputInstance, "timeout.cnf");
  while ((c = getopt(iArgc_, aArgv_, "i:p:l:#:s:t:r:u:f:o:c:h?v")) != -1) {
    switch(c) {
    case 'v':
    case '?':
    case 'h':
      printHelpMessageAndExit();
    case 'p':
      sscanf(optarg, "%d", &iPreprocessLevel);
      if (iPreprocessLevel < 0 || iPreprocessLevel > 3) {
	printHelpMessageAndExit();
      }
      break;
    case 'c':
      sscanf(optarg, "%d", &iOutputPreprocessLevel);
      if (iOutputPreprocessLevel < 0 || iOutputPreprocessLevel > 3) {
	printHelpMessageAndExit();
      }
      break;
    case 'i':
      if (optarg[0] == 'n') {
	iIterationBound = -1;
      }
      else {
	sscanf(optarg, "%d", &iIterationBound);
	if (iIterationBound < 1) {
	  printHelpMessageAndExit();
	}
      }
      break;
    case 'l':
      sscanf(optarg, "%d", &iWork);
      if (iWork < 0) {
	printHelpMessageAndExit();
      }
      xSATSolver.vSetLearnOrder(iWork);
      break;
    case '#':
      if (optarg[0] == 'a') {
	xSATSolver.vSetSolutionLimit(0);
      }
      else if (optarg[0] == 'c') {
	xSATSolver.vSetFindAll(0);
      }
      else {
	sscanf(optarg, "%d", &iWork);
	if (iWork < 1) {
	  printHelpMessageAndExit();
	}	
	xSATSolver.vSetSolutionLimit(iWork);
      }
      break;
    case 's':
      sscanf(optarg, "%d", &iRandomSeed);
      if (iRandomSeed < 1) {
	printHelpMessageAndExit();
      }
      break;
    case 'r':
      if (optarg[0] != 'n') {
	sscanf(optarg, "%d", &iWork);
	if (iWork <= 0) {
	  printHelpMessageAndExit();
	}
	xSATSolver.vSetRestartInterval(iWork);
      }
      break;
    case 't':
      if (optarg[0] == 'n') {
	xSATSolver.vSetNoTimeLimit(1);
	bNoTimeout = 1;
      }
      else {
	sscanf(optarg, "%d", &iWork);
	if (iWork < 0) {
	  printHelpMessageAndExit();
	}
	xSATSolver.vSetTimeout(iWork);
      }
      break;
    case 'f':
      sscanf(optarg, "%f", &fWork);
      if (fWork <0.0 || fWork > 1.0) {
	printHelpMessageAndExit();
      }
      xSATSolver.vSetFudgeFactor(fWork);
      break;
    case 'u':
      if (optarg[0] == 'n') {
	xSATSolver.vSetPrintStack(0);
      }
      else {
	sscanf(optarg, "%d", &iWork);
	if (iWork < 1) {
	  printHelpMessageAndExit();
	}
	xSATSolver.vSetPrintStackPeriod(iWork);
      }
      break;
    case 'o':
      int len = strlen(optarg);
      if (len >= 512) {
	printHelpMessageAndExit();
      }
      if (len == 1 && optarg[0] == 'n') {
	bOutputInstance = 0;
      }
      else {
	strcpy(aOutputInstance, optarg);
      }
      break;
    } // switch
  }

  if (optind >= iArgc_) {
    printHelpMessageAndExit();
  }
  cout << "c Instance is: " << aArgv_[optind] << endl; 
  if (!xInstance.bReadDimacs(aArgv_[optind]))
    exit(3);
  Random::vInitRandom(iRandomSeed);
  cout << "c Random number seed: " << iRandomSeed << endl;

  if (iPreprocessLevel) {
    cout << "c Pre-processing level: " << iPreprocessLevel << endl;
  }
  if (!bNoTimeout && bOutputInstance) {
    if (iOutputPreprocessLevel) {
      cout << "c Post-processing level: " << iOutputPreprocessLevel << endl;
    }
    cout << "c Output instance filename: " << aOutputInstance << endl;
  }
  if ((iPreprocessLevel || iOutputPreprocessLevel) && iIterationBound >= 0) {
    cout << "c Pre/Post-Processing iteration bound: " << iIterationBound << endl;
  }

  boolean bFailed;

  xSATSolver.vOutputStatusUpdateInterval();
  if (xSATSolver.bPreprocess(iPreprocessLevel,iIterationBound)) {
    cout << "c Determined during preprocessing:\n";
    cout << "UNSAT\n";
  }
  else {
    char* aCount;
    xSATSolver.vOutputWarnings();
    relsat_enum eResult = xSATSolver.eSolve();
    cout << "c Solution phase stats: " << endl;
    cout << "c   Variable Assignments        : " << xSATSolver.iVariablesLabeled() << endl;
    cout << "c   Branch Selections           : " << xSATSolver.iBranchSelections() << endl;
    cout << "c   Contradictions Discovered   : " 
	 << xSATSolver.iContradictionsDiscovered() 
	 << endl;
    cout << "c   Seconds Elapsed (real time) : " << xSATSolver.iElapsedSeconds() << endl;
    switch (eResult) {
    case SAT:
      if (xSATSolver.bIsCounting()) {
	cout << "Number of solutions: "; 
	aCount = xSATSolver.pSolutionCount()->aToString(); 
	cout << aCount << endl;
	delete [] aCount;
      }
      cout << "SAT\n";
      break;
    case UNSAT:
      cout << "UNSAT\n";
      break;
    case TIMEOUT:
      xSATSolver.vIncorporateLearnedClauses();
      if (bOutputInstance && xSATSolver.bPreprocess(iOutputPreprocessLevel,
						    iIterationBound)) { 
	cout << "c Determined during preprocessing after a timeout:\n";
	cout << "UNSAT\n";
      }
      else {
	if (bOutputInstance) {
	  // output instance and its learned clauses.
	  cout << "c Writing instance into " << aOutputInstance << "..." << flush;
	  ofstream xNewInstance;
	  xNewInstance.open(aOutputInstance, ios::out);
	  if (!xNewInstance) {
	    cout << "Failed to open output instance!" << endl;
	    exit(2);
	  }
	  xInstance.vOutputDimacs(xNewInstance);
	  cout << "..done." << endl;
	}
	cout << "TIME LIMIT EXPIRED" << endl;
      }
    }
  }
}
